<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\User;
use App\Models\UserLocation;
use App\Models\DepartmentUser;
use App\Models\Role;
use Illuminate\Support\Facades\Hash;
use App\Models\LoginActivity;
use App\Models\ActivityLog;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;
use App\Models\UserAssignment;

class UserController extends Controller
{
    public function index()
    {
        $users = User::with('roles')->get();
        $roles = Role::pluck('name');

       // print_r($users);die();
        return view('admin.users.index', compact('users', 'roles'));
    }

// Store User
public function store(Request $request)
{
    $request->validate([
        'name'       => 'required|string',
        'email'      => 'required|email|unique:users,email',
        'password'   => 'required|min:6',
        'role'       => 'required|exists:roles,id',
        'user_type'  => 'required',
        'department' => 'required|array|min:1',
        'department.*' => 'exists:departments,id',
        'status'     => 'required',
        'assigned_locations' => 'array',
        'assigned_locations.*' => 'exists:locations,id',
        'assigned_users' => 'array',
        'assigned_users.*' => 'exists:users,id',
    ]);

    DB::transaction(function () use ($request) {
        $role = Role::findOrFail($request->role);

        $user = User::create([
            'role_id'              => $role->id,
            'user_type'            => $request->user_type,
            'reporting_manager_id' => $request->reporting_manager_id,
            'name'                 => $request->name,
            'email'                => $request->email,
            'password'             => Hash::make($request->password),
            'phone'                => $request->phone,
            'joining_date'         => $request->joining_date,
            'access_expires_at'    => $request->access_expires_at,
            'status'               => $request->status,
        ]);

        $user->syncRoles([$role->id]);

        // ✅ Assign departments
        $user->departments()->sync($request->department);

        // ✅ Assign locations
        if ($request->filled('assigned_locations')) {
            $user->locations()->sync($request->assigned_locations);
        }

        // ✅ Assign users (same department)
        if ($request->filled('assigned_users')) {
            $assignedUsers = [];
            foreach ($request->assigned_users as $assignedUserId) {
                $assignedUser = User::findOrFail($assignedUserId);

                // Optional: check if assigned user is in one of the user's departments
                if (!$assignedUser->departments->pluck('id')->intersect($request->department)->count()) {
                    throw new \Exception("Assigned user must belong to one of the selected departments.");
                }

                $assignedUsers[] = [
                    'user_id' => $user->id,
                    'assigned_user_id' => $assignedUserId,
                    'department_id' => $assignedUser->department_id,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
            if ($assignedUsers) {
                \DB::table('user_assignments')->insert($assignedUsers);
            }
        }
    });

    return response()->json(['status' => true, 'message' => 'User created successfully']);
}


// Update User
public function update(Request $request, User $user)
{
    $request->validate([
        'name'       => 'required|string',
        'email'      => 'required|email|unique:users,email,' . $user->id,
        'password'   => 'nullable|min:6',
        'role'       => 'required|exists:roles,id',
        'user_type'  => 'required',
        'department' => 'required|array|min:1',
        'department.*' => 'exists:departments,id',
        'status'     => 'required',
        'assigned_locations' => 'array',
        'assigned_locations.*' => 'exists:locations,id',
        'assigned_users' => 'array',
        'assigned_users.*' => 'exists:users,id',
    ]);

    DB::transaction(function () use ($request, $user) {
        $role = Role::findOrFail($request->role);

        $user->update([
            'role_id'              => $role->id,
            'user_type'            => $request->user_type,
            'reporting_manager_id' => $request->reporting_manager_id,
            'name'                 => $request->name,
            'email'                => $request->email,
            'phone'                => $request->phone,
            'joining_date'         => $request->joining_date,
            'access_expires_at'    => $request->access_expires_at,
            'status'               => $request->status,
        ]);

        if ($request->filled('password')) {
            $user->update(['password' => Hash::make($request->password)]);
        }

        $user->syncRoles([$role->id]);

        // ✅ Sync departments
        $user->departments()->sync($request->department);

        // ✅ Sync locations
        $user->locations()->sync($request->assigned_locations ?? []);

        // ✅ Sync assigned users (clear old first)
        \DB::table('user_assignments')->where('user_id', $user->id)->delete();
        if ($request->filled('assigned_users')) {
            $assignedUsers = [];
            foreach ($request->assigned_users as $assignedUserId) {
                $assignedUser = User::findOrFail($assignedUserId);

                if (!$assignedUser->departments->pluck('id')->intersect($request->department)->count()) {
                    throw new \Exception("Assigned user must belong to one of the selected departments.");
                }

                $assignedUsers[] = [
                    'user_id' => $user->id,
                    'assigned_user_id' => $assignedUserId,
                    'department_id' => $assignedUser->department_id,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }
            if ($assignedUsers) {
                \DB::table('user_assignments')->insert($assignedUsers);
            }
        }
    });

    return response()->json(['status' => true, 'message' => 'User updated successfully']);
}




public function edit(User $user)
{
   return view('admin.users.edit', [
                'user' => $user,
        'role' => $user->roles->pluck('id')->first(),

        'department_ids' => $user->departments()->pluck('departments.id'),

        'reporting_manager_id' => $user->reporting_manager_id,

        'assigned_locations' => $user->locations()->pluck('locations.id'),

        'assigned_users' => $user->assignedUsers()->pluck('users.id'),
       ]);
    // return response()->json([
    //     'user' => $user,
    //     'role' => $user->roles->pluck('id')->first(),

    //     'department_ids' => $user->departments()->pluck('departments.id'),

    //     'reporting_manager_id' => $user->reporting_manager_id,

    //     'assigned_locations' => $user->locations()->pluck('locations.id'),

    //     'assigned_users' => $user->assignedUsers()->pluck('users.id'),
    // ]);
}


// public function update(Request $request, User $user)
// {
//     try {
//         $request->validate([
//             'name'       => 'required|string',
//             'email'      => 'required|email|unique:users,email,' . $user->id,
//             'password'   => 'nullable|min:6',
//             'role'       => 'required|exists:roles,id',
//             'user_type'  => 'required',
//              'department'   => 'required|array|min:1',
//             'department.*' => 'exists:departments,id',
//             'status'     => 'required',

//             'assigned_locations' => 'array',
//             'assigned_locations.*' => 'exists:locations,id',

//             'assigned_users' => 'array',
//             'assigned_users.*' => 'exists:users,id',
//         ]);

//         DB::transaction(function () use ($request, $user) {

//             $role = Role::findOrFail($request->role);

//             $user->update([
//                 'role_id'              => $role->id,
//                 'user_type'            => $request->user_type,
//                 'reporting_manager_id' => $request->reporting_manager_id,
//                 'name'                 => $request->name,
//                 'email'                => $request->email,
//                 'phone'                => $request->phone,
//                 'joining_date'         => $request->joining_date,
//                 'access_expires_at'    => $request->access_expires_at,
//                 'status'               => $request->status,
//             ]);

//             if ($request->filled('password')) {
//                 $user->update([
//                     'password' => Hash::make($request->password),
//                 ]);
//             }

//             $user->syncRoles([$role->id]);

//             /* =========================
//                CLEAR OLD ASSIGNMENTS
//             ========================= */
//             UserLocation::where('user_id', $user->id)->delete();
//             UserAssignment::where('user_id', $user->id)->delete();

//             /* =========================
//                RE-ASSIGN LOCATIONS
//             ========================= */
//             if ($request->filled('assigned_locations')) {
//                 foreach ($request->assigned_locations as $locationId) {
//                     UserLocation::create([
//                         'user_id'     => $user->id,
//                         'location_id' => $locationId,
//                     ]);
//                 }
//             }

//             /* =========================
//                RE-ASSIGN USERS (SAME DEPT)
//             ========================= */
//             if ($request->filled('assigned_users')) {
//                 foreach ($request->assigned_users as $assignedUserId) {

//                     $assignedUser = User::find($assignedUserId);

//                     if ($assignedUser->department_id !== $user->department_id) {
//                         throw new \Exception('Assigned user must belong to same department');
//                     }

//                     UserAssignment::create([
//                         'user_id'          => $user->id,
//                         'assigned_user_id' => $assignedUserId,
//                         'department_id'    => $user->department_id,
//                     ]);
//                 }
//             }
//         });

//         return response()->json([
//             'status'  => true,
//             'message' => 'User updated successfully'
//         ]);

//     } catch (\Throwable $e) {
//         return response()->json([
//             'status'  => false,
//             'message' => $e->getMessage(),
//         ], 500);
//     }
// }



    public function destroy(User $user)
    {
        if ($user->id === auth()->id()) {
            return response()->json([
                'status' => false,
                'message' => 'You cannot delete yourself'
            ], 403);
        }

        $user->delete();

        return response()->json([
            'status' => true,
            'message' => 'User deleted successfully'
        ]);
    }

    public function loginHistory(User $user)
    {
        $activities = LoginActivity::where('user_id', $user->id)
            ->latest('login_at')
            ->get();

        return view('admin.users.login-history', compact('user', 'activities'));
    }

    public function deleteLoginActivity(LoginActivity $activity)
    {
        $activity->delete();

        return response()->json([
            'status' => true,
            'message' => 'Login activity deleted'
        ]);
    }
    public function deleteAllLoginHistory(User $user)
    {
        $user->loginActivities()->delete();

        return response()->json([
            'status' => true,
            'message' => 'All login history deleted'
        ]);
    }

    public function deleteAllActivityLog(User $user)
    {
        $user->activityLog()->delete();

        return response()->json([
            'status' => true,
            'message' => 'All login history deleted'
        ]);
    }


    public function show($id)
    {
        $user = User::with(['role', 'department'])->findOrFail($id);

        $activities = ActivityLog::where('causer_id', $id)
            ->latest()
            ->limit(50)
            ->get();

        $loginLogs = LoginActivity::where('user_id', $id)
            ->latest()
            ->limit(50)
            ->get();


        $activities = LoginActivity::where('user_id', $user->id)
            ->latest('login_at')
            ->get();

        return view('admin.users.profile', compact('user', 'activities', 'loginLogs'));
    }
    public function active()
    {
        return DataTables::of(
            User::where('status', 'active')
        )->make(true);
    }

    public function inactive()
    {
        return DataTables::of(
            User::where('status', 'inactive')
        )->make(true);
    }


// public function getDepartmentUsers($departmentId)
// {
//     return User::where('department_id', $departmentId)->get(['id', 'name']);
// }
public function getDepartmentUsers(Request $request)
{
    $departmentIds = (array) $request->department_ids;

    return response()->json(
        User::whereHas('departments', function ($q) use ($departmentIds) {
            $q->whereIn('departments.id', $departmentIds);
        })
        ->when($request->user_id, fn ($q) => $q->where('id', '!=', $request->user_id))
        ->select('id', 'name')
        ->get()
    );
}



// controller
public function getRoleDepartments(Role $role)
{
    print_r($role->departments());die();
    return response()->json(
        $role->departments()->select('id', 'name')->get()
    );
}



}
